﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;

namespace hohl.DiffUpdate
{
    /// <summary>
    /// Führt ein Update aus.
    /// </summary>
    public class Update
    {
        #region Ereignisse
        public delegate void UpdateEventHandler(string text);
        /// <summary>
        /// Wird aufgerufen wenn ein Dateidownload gestartet wurde.
        /// </summary>
        public UpdateEventHandler DownloadStarted;
        /// <summary>
        /// Wird aufgerufen wenn ein Dateidownload abgeschlossen wurde.
        /// </summary>
        public UpdateEventHandler DownloadComplete;
        /// <summary>
        /// Wird aufgerufen wenn eine Datei geprüft wird.
        /// </summary>
        public UpdateEventHandler CheckFile;
        /// <summary>
        /// Wird aufgerufen wenn die Datei aktualisiert wurde.
        /// </summary>
        public UpdateEventHandler FileUpdated;
        /// <summary>
        /// Wird aufgerufen wenn der Update Vorgang beendet wurde.
        /// </summary>
        public UpdateEventHandler UpdateProcessComplete;

        public delegate bool UpdateErrorHandler(string text);
        /// <summary>
        /// Wird aufgerufen wenn ein Fehler während des Dateidownloads aufgetreten ist.
        /// Wenn true zurückgegeben wird, wird das Update fortgesetzt.
        /// </summary>
        public UpdateErrorHandler DownloadError;
        #endregion

        /// <summary>
        /// Trennzeichen zum auflösen der Update Informations Zeile
        /// </summary>
        private char[] seperators = { ';' };

        /// <summary>
        /// Temporäres Verzeichnis zum zwischenspeichern der alten Dateien.
        /// </summary>
        private string tempDir;

        /// <summary>
        /// Verzeichnis in dem das Update ausgeführt werden soll.
        /// </summary>
        protected string rootPath;

        /// <summary>
        /// Datei mit den Information des Updateablaufes
        /// </summary>
        protected StreamReader updateFile;

        /// <summary>
        /// Bereitet einen neuen Update Prozess vor.
        /// </summary>
        /// <param name="file">Datei mit Updateinformationen</param>
        /// <param name="path">Pfad in welchen die Datei installiert werden soll</param>
        public Update(StreamReader file, string path)
        {
            this.rootPath = path;
            this.updateFile = file;
        }

        /// <summary>
        /// Startet den Updatevorgang.
        /// </summary>
        public void StartUpdate()
        {
            // Alle vorhandenen Dateien ins Temporäre Verzeichnis sichern
            tempDir = Path.Combine(Path.GetPathRoot(rootPath), "tempBackup");
            if (Directory.Exists(rootPath))
                Directory.Move(rootPath, tempDir);


            // Das Installiationsverzeichnis leeren
            Trace.WriteLine(rootPath);
            Directory.CreateDirectory(rootPath);

            // Dateien aktualisieren
            while (!updateFile.EndOfStream)
            {
                string line = updateFile.ReadLine();
                if (line.Split(seperators).Length != 3)
                    throw new IOException("Error in Update Information Script");
                string file = line.Split(seperators)[0];
                string checksum = line.Split(seperators)[1];
                string downloadUrl = line.Split(seperators)[2];
                UpdateFile(file, checksum, downloadUrl);
            }

            // Temporäre Dateien wieder löschen
            if (Directory.Exists(tempDir))
                Directory.Delete(tempDir, true);

            // Ereignis aufrufen
            if (UpdateProcessComplete != null)
                UpdateProcessComplete(null);
        }

        /// <summary>
        /// Aktualisiert die angegebene Datei.
        /// </summary>
        /// <param name="file">Zu aktualisierende Datei (Kann auch relativen Pfad beinhalten)</param>
        /// <param name="checksum">Prüfsumme der zu aktualisierenten Datei</param>
        /// <param name="url">Download URL falls die Datei benötigt wird</param>
        protected void UpdateFile(string file, string checksum, string url)
        {
            if (File.Exists(Path.Combine(tempDir, file)) && CheckMD5File(Path.Combine(tempDir, file), checksum))
            {
                // Wenn die vorhandene Datei noch aktuell ist, kopiere sie in das neue Verzeichnis
                File.Copy(Path.Combine(tempDir, file), Path.Combine(rootPath, file));
            }
            else
            {
                // Sollte die Datei noch nicht exestieren oder veraltet sein, so lade sie nun herunter
                try
                {
                    // Download gestartet:
                    if (DownloadStarted != null)
                        DownloadStarted(url);
                    // Downloade Datei
                    Download.GetFile(url, Path.Combine(rootPath, file));
                    // Download erfolgreich:
                    if (DownloadComplete != null)
                        DownloadComplete(url);
                    // Datei erfolgreich aktualisiert:
                    if (FileUpdated != null)
                        FileUpdated(file);
                }
                catch
                {
                    // Fehler beim download:
                    if (DownloadError != null)
                        DownloadError(url);
                }
            }
        }

        /// <summary>
        /// Prüft ob die Prüfsumme (MD5) mit der angegeben Datei übereinstimmt.
        /// </summary>
        /// <param name="Dateipfad">Zu prüfende Datei</param>
        /// <param name="Checksumme">Prüfsumme</param>
        /// <returns>true wenn die Prüfsummen übereinstimmen.</returns>
        protected bool CheckMD5File(string file, string checksum)
        {
            // Prüfe Datei auf aktualität.
            if (CheckFile != null)
                CheckFile(file);
            
            // Datei einlesen und MD5 Prüfsumme berechnen
            FileStream fstream = File.OpenRead(file);
            System.Security.Cryptography.MD5 md5 = new System.Security.Cryptography.MD5CryptoServiceProvider();
            byte[] md5Hash = md5.ComputeHash(fstream);
            fstream.Close();

            // ByteArray in String umwandeln
            string filesum = BitConverter.ToString(md5Hash).Replace("-", "");

            if (filesum.ToLower() == checksum.ToLower())
                return true;
            else
                return false;
        }
    }
}
